/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/

#ifndef __ASN1CUTCTIME_H__
#define __ASN1CUTCTIME_H__

#include "ASN1CTime.h"

//////////////////////////////////////////////////////////////////////
//
// ASN1CUTCTime
// 
// ISO 8601 time and date for ITU-T X.680. 
//
// Author Artem Bolgar.
// version 2.13   11 Jul, 2003 (2038 year problem fixed + some other fixes)
// version 1.11   3 Aug, 2002
//

/**
 * @file ASN1CUTCTime.h
 * ASN1CUTCTime control class definition.
 */
/**
 * @addtogroup asn1ctrl 
 * @{
 */
/**
 * @addtogroup asn1ctime 
 * @{
 */

#ifndef _NO_UTILS_CLASSES
/**
 * ASN.1 UTCTime control class. 
 * The ASN1CUTTime class is derived from the ASN1CTime base class. It used as
 * the bass class for generated control classes for the ASN.1 Universal Time
 * ([UNIVERSAL 23] IMPLICIT VisibleString) type. This class provides utility
 * methods for operating on the time information referenced by the generated
 * class. This class can also be used inline to operate on the time within
 * generated time string elements in a SEQUENCE, SET, or CHOICE construct. The
 * string generally is encoded according to ISO 8601 format with some
 * exceptions (see X.680).
 */
class EXTERNRT ASN1CUTCTime : public ASN1CTime {
 protected:
   ASN1CUTCTime(char*& buf, int bufSize, ASN1BOOL useDerRules = FALSE); 
   ASN1CUTCTime(ASN1UTCTime& buf, ASN1BOOL useDerRules = FALSE); 

 public:
  /**
   * This constructor creates a time string from a buffer.
   *
   * It does not deep-copy the data, it just assigns the passed array to an
   * internal reference variable. The object will then directly operate on the
   * given data variable.
   *
   * @param msgBuf       Reference to an ASN1MEssage buffer derived object (for
   *                       example, an ASN1BEREncodeBuffer).
   * @param buf          Reference to a pointer to a time string buffer.
   * @param bufSize      Size of passed buffer, in bytes.
   * @param useDerRules  Use the Distinguished Encoding Rules to encode or
   *                       decode the value,
   */
   ASN1CUTCTime(ASN1MessageBufferIF& msgBuf, char*& buf, int bufSize,
      ASN1BOOL useDerRules = FALSE); 

  /**
   * This constructor creates a time string suing the ASN1UTCTime argument. c
   * The constructor does not deep-copy the variable, it assigns a referene to
   * it to an internal variable. The object will then directly operate on the
   * given data variable. This form of the constructor is used with a
   * compiler-generated time string variable.
   *
   * @param msgBuf       Reference to an ASN1MEssage buffer derived object (for
   *                       example, an ASN1BEREncodeBuffer).
   * @param buf          Reference to a time string structure.
   * @param useDerRules  Use the Distinguished Encoding Rules to encode or
   *                       decode the value,
   */
   ASN1CUTCTime(ASN1MessageBufferIF& msgBuf, ASN1UTCTime& buf,
      ASN1BOOL useDerRules = FALSE); 

   ASN1CUTCTime(ASN1Context& ctxt, char*& buf, int bufSize,
      ASN1BOOL useDerRules = FALSE); 

   ASN1CUTCTime(ASN1Context& ctxt, ASN1UTCTime& buf,
      ASN1BOOL useDerRules = FALSE); 

   // copy ctor

   ASN1CUTCTime (const ASN1CUTCTime& original) :
      ASN1CTime (original) {}

  /**
   * This method sets the year component of the time value.
   *
   * The year parameter can be either the two last digits of the year
   * (00 - 99) or the full four digits (0 - 9999). Note: the getYear method
   * returns the year in the full four digits, independent of the format of the
   * year parameter used in this method.
   *
   * @param year_        Year component (full four digits or only last two
   *                       digits).
   * @return             Completion status of operation:
   *                       - 0 (ASN_OK) = success,
   *                       - negative return value is error.
   */
   int setYear(int year_);

  /**
   * Converts time_t to time string.
   *
   * @param time         time to convert,
   * @param diffTime     TRUE means the difference between local time and UTC
   *                       will be calculated; in other case only local time
   *                       will be stored.
   * @return             Completion status of operation:
   *                       - 0 (ASN_OK) = success,
   *                       - negative return value is error.
   */
   int setTime(time_t time, ASN1BOOL diffTime);

  /** 
   * This method sets the UTC flag state.
   *
   * If the UTC flag is TRUE, then the time is a UTC time and symbol 'Z' is
   * added to the end of the string. Otherwise, it is a local time.
   *
   * @param utc          UTC flag state.
   * @return             Completion status of operation:
   *                       - 0 (ASN_OK) = success,
   *                       - negative return value is error.
   */
   int setUTC (ASN1BOOL utc);

  /**
   * Clears out time string.
   *
   * @param             - none
   */
   void clear();

  /**
   * Parses the given time string.  The string is assumed to be in standard 
   * UTC time format.
   *
   * @param string      UTC time string to be parsed.
   * @return             Completion status of operation:
   *                       - 0 (ASN_OK) = success,
   *                       - negative return value is error.
   */
   int parseString(ASN1ConstCharPtr string);

   const ASN1CUTCTime& operator = (const ASN1CUTCTime& tm) {
      return (ASN1CUTCTime&) ASN1CTime::operator = (tm);
   }

 protected:
   /*
   * Compiles new time string accoring X.680 (clause 42) and ISO 8601.
   * Returns ASN_OK, if successful, or error code, if error.
   */
   int compileString();

   /* 
   * Gets fraction of second, 0..9. 
   * Returns: Zero.
   */
   int getFraction();
   
   /*
   * This method is not applicable for ASN1CUTCTime.
   * Returns ASN_E_NOTSUPP.
   */
   int setFraction(int fraction);

} ;
#else
typedef class _ASN1CUTCTime : public _ASN1CTime {
 protected:
   _ASN1CUTCTime(char*& buf, int bufSize, ASN1BOOL useDerRules) {}
   _ASN1CUTCTime(ASN1VisibleString& buf,  ASN1BOOL useDerRules) {}
 public:
   _ASN1CUTCTime (ASN1MessageBufferIF& msgBuf, char*& buf, int sz,
      ASN1BOOL useDerRules = FALSE) :
      _ASN1CTime (msgBuf, buf, sz, useDerRules) {} 
   _ASN1CUTCTime (ASN1MessageBufferIF& msgBuf, ASN1VisibleString& buf,
      ASN1BOOL useDerRules = FALSE) :
      _ASN1CTime (msgBuf, buf, useDerRules) {} 
   _ASN1CUTCTime(ASN1Context& ctxt, char*& buf, int bufSize, 
      ASN1BOOL useDerRules) : _ASN1CTime (ctxt, buf, bufSize, useDerRules) {}  
   _ASN1CUTCTime(ASN1Context& ctxt, ASN1VisibleString& buf, 
      ASN1BOOL useDerRules) : _ASN1CTime (ctxt, buf, bufSize, useDerRules) {}  
} ASN1CUTCTime;
#endif // _NO_UTILS_CLASSES
#endif //__ASN1CUTCTIME_H__

/**
 * @} 
 */
/**
 * @} 
 */
